#include <stdafx.h>
#include "MovingObject.h"

#include <math.h>
#include <process.h>
#include <algorithm>
#include <fstream>

/*****************************************************************************/
/*                                                                           */
/*                 M O V I N G   O B J E C T   B A S E                       */
/*                                                                           */
/*****************************************************************************/

unsigned int MovingObjectBase::sm_NextID = 1;

MovingObjectBase::MovingObjectBase()
: m_ID(-1), m_Typ(NONE), m_TimeStamp(0), m_Size(1), m_Touched(true)
{
	Reset();
}

MovingObjectBase::MovingObjectBase(ObjectType typ, double ts, Point centre)
: m_ID(sm_NextID++), m_Typ(typ), m_TimeStamp(ts), m_Size(1), m_Centre(centre), m_LastCentre(centre), m_Touched(true)
{
	Reset();
}

void MovingObjectBase::Reset() {
	m_DxSum = m_DySum = m_DCount = 0;
	m_Speed.m_x = m_Speed.m_y = 0;
	m_InterP.m_x = m_InterP.m_y = 0;
	m_Dist = m_Time2Impact = 0;
	m_ClosestDist = 999999.0;
	m_DeadlyObject = false;
}

// look if estimated position and given position / type / size match: if so update object
bool MovingObjectBase::UpdateMatchingObject(double ts, ObjectType typ, int x, int y, int /*scaleFactor*/, int /*subType*/) {
	// we updated this already or wrong typ
	if( m_Touched || ( (m_Typ & typ) == 0) ) return false;

	// check if it might be this object
	double t = ts - m_TimeStamp;

	// check for wrap around
	double cx = m_LastCentre.m_x, cy = m_LastCentre.m_y;
	if( (cx - x) > MAX_DX_RES/2 ) {
		cx -= MAX_DX_RES;
	}
	else if( (cx - x) < -MAX_DX_RES/2 ) {
		cx += MAX_DX_RES;
	}
	if( (cy - y) > MAX_DY_RES/2 ) {
		cy -= MAX_DY_RES;
	}
	else if( (cy - y) < -MAX_DY_RES/2 ) {
		cy += MAX_DY_RES;
	}

	// if enough points we can determine speed good enough
	if( m_DCount > 7 ) {
		// distance to estimated position in x
		double dx = x - (cx + t * m_Speed.m_x);
		// distance to estimated position in y
		double dy = y - (cy + t * m_Speed.m_y);

		// squared distance of estimated position to given position
		double dist = dx*dx+dy*dy;

		// distance to estimated position small enough: must be this object
		if( dist < 18 ) {
			Update(ts, x, y);
			return true;
		}
	}
	// otherwise just use current position
	else {
		// distance to last position in x
		int dx = (int)(x - cx);
		// distance to last position in y
		int dy = (int)(y - cy);

		// squared distance of current position to given position
		int dist = dx*dx+dy*dy;

		// distance to last position small enough: must be this object
		if( dist < 130 ) {
			Update(ts, x, y);
			return true;
		}
	}
	return false;
}

void MovingObjectBase::Update(double ts, int x, int y) {
	// check for wrap around
	if( (m_LastCentre.m_x - x) > MAX_DX_RES/2 ) {
		m_LastCentre.m_x -= MAX_DX_RES;
	}
	else if( (m_LastCentre.m_x - x) < -MAX_DX_RES/2 ) {
		m_LastCentre.m_x += MAX_DX_RES;
	}
	if( (m_LastCentre.m_y - y) > MAX_DY_RES/2 ) {
		m_LastCentre.m_y -= MAX_DY_RES;
	}
	else if( (m_LastCentre.m_y - y) < -MAX_DY_RES/2 ) {
		m_LastCentre.m_y += MAX_DY_RES;
	}

	// update timestamp, position and summs
	double dt = ts - m_TimeStamp;
	m_TimeStamp = ts;

    // substract last speed to get previous position to determine correct offset
	double dx = x - m_LastCentre.m_x;
	m_LastCentre.m_x = x;
	double dy = y - m_LastCentre.m_y;
	m_LastCentre.m_y = y;

	// calculate speed
	if(m_Typ == UFO) {
		if( ( (long)m_DCount % 8) == 0) {
			m_DxSum = 0;
			m_DySum = 0;
//			m_DCount = 0;
		}
		m_DxSum += dx;
		m_DySum += dy;
		m_DCount += dt;

        double rest = ((long)m_DCount % 8);
		if( rest > 4) {
			m_Speed.m_x = m_DxSum / rest;
			m_Speed.m_y = m_DySum / rest;
		}
	}
	else if(m_DCount < 8) {
		m_DxSum += dx;
		m_DySum += dy;
		m_DCount += dt;
		m_Speed.m_x = m_DxSum / m_DCount;
		m_Speed.m_y = m_DySum / m_DCount;
	}
	else {
		// check if speed okay
		if( (fabs(dx - m_Speed.m_x) > 1) || (fabs(dy - m_Speed.m_y) > 1) ) {
			m_DxSum = dx;
			m_DySum = dy;
			m_DCount = dt;
			m_Speed.m_x = m_DxSum / m_DCount;
			m_Speed.m_y = m_DySum / m_DCount;
		}
		else {
			m_DCount += dt;
		}
	}

    // move position to next estimate!
    // this way calculations and matching is easier!
    m_Centre.m_x = m_LastCentre.m_x + m_Speed.m_x;
    m_Centre.m_y = m_LastCentre.m_y + m_Speed.m_y;

	if( (m_Typ == FF) && (m_DCount == 8) ) {
		Statistics *stat = Statistics::GetStatistics();
		stat->m_SHOT_SPEED = (100.0 * stat->m_SHOT_SPEED + sqrt(m_Speed.m_x * m_Speed.m_x + m_Speed.m_y * m_Speed.m_y)) / 101.0;
	}
	m_Touched = true;
}

// determine fly-by and other dynamic data
void MovingObjectBase::CalcDynamicData(double shotAngle) {
	// no need to calc this for my own shots!
	if(m_Typ == FF) return;

	// 1. direct distance
	m_Dist = max(0.0, sqrt(m_Centre.m_x * m_Centre.m_x + m_Centre.m_y * m_Centre.m_y) - m_Size);

	// 2. distance of closest "fly-by" point
	// (set to 0,0 if flies not toward me or if I can not determine point)
	m_ClosestDist = 999999.0;
	if( (m_Speed.m_x != 0) && (m_Speed.m_y != 0) ) {
		Line astroV(m_Centre, Point(m_Speed.m_x, m_Speed.m_y) );
		// normal vector of astro is my "direction"
		Line shipV(0, 0, -m_Speed.m_y, m_Speed.m_x );
		if( astroV.intersection(shipV, m_InterP) ) {
			// calc direction
			double f = astroV.Factor(m_InterP);
			// if it flies towards me
			if(f > 0.0) {
				// calc closes distance to me
				m_ClosestDist = max(0.0, sqrt(m_InterP.m_x * m_InterP.m_x + m_InterP.m_y * m_InterP.m_y) - m_Size);
			}
		}
	}
	// could not determine or already passed me!
	if(m_ClosestDist == 999999.0) {
		m_InterP.m_x = m_InterP.m_y = 0.0;
	}

	m_DeadlyObject = false;
	m_Time2Impact = 9999;
	if(m_ClosestDist < 19) {
		double speed = max(0.125, sqrt(m_Speed.m_x * m_Speed.m_x + m_Speed.m_y * m_Speed.m_y));
		m_Time2Impact = m_Dist / speed;
		if( (m_Typ == SHOT) || (m_Time2Impact < 70) ) {
			m_DeadlyObject = true;
		}
	}
}


/*****************************************************************************/
/*                                                                           */
/*                 S H O T A B L E   O B J E C T                             */
/*                                                                           */
/*****************************************************************************/

ShootableObject::ShootableObject() 
: m_Sf(1), m_MaxShots(1)
{
	m_EstimatedBlowTime = m_AimAngle = m_AimAngleOffset = m_Time2Turn = m_Time2Wait = m_Time2Hit = m_Distance2Aim = 0.0;
    m_Time2TurnLast = m_Time2WaitLast = m_Time2HitLast = 0;
	m_Shots = m_FasterTurnToHit = m_FasterTurnToHitLast = 0;
    m_Time2Wait = -1;
}


ShootableObject::ShootableObject(ObjectType typ, double ts, Point centre, int sf) 
: MovingObjectBase(typ, ts, centre), m_Sf(sf), m_MaxShots(1)
{
	m_EstimatedBlowTime = m_AimAngle = m_AimAngleOffset = m_Time2Turn = m_Time2Hit = m_Distance2Aim = 0.0;
    m_Time2TurnLast = m_Time2WaitLast = m_Time2HitLast = 0;
	m_Shots = m_FasterTurnToHit = m_FasterTurnToHitLast = 0;
    m_Time2Wait = -1;
}

// look if estimated position and given position / type / size match: if so update object
bool ShootableObject::UpdateMatchingObject(double ts, ObjectType typ, int x, int y, int scaleFactor, int /*subType*/) {
	// check subtype and size
	bool bret = (m_Sf == scaleFactor);
	if(bret) {
		// match: check other things
		bret = MovingObjectBase::UpdateMatchingObject(ts, typ, x, y, scaleFactor, 0);
	}
	return bret;
}


// calc hit point with given shot angle
void ShootableObject::CalcAimPoint(double shotAngle, double deltaT, Point &aimPoint, double &distance2Aim, double &aimAngle, double &aimAngleOffset, double &time2Hit) {
	static Statistics *stat = Statistics::GetStatistics();

	double newCentreX = NormalizeX(m_Centre.m_x + deltaT * m_Speed.m_x);
	double newCentreY = NormalizeY(m_Centre.m_y + deltaT * m_Speed.m_y);
	
	// init aim point
	aimPoint.m_x = newCentreX; aimPoint.m_y = newCentreY;
	// init time to hit, i.e. time bullet flies towards aim point
	time2Hit = sqrt(newCentreX * newCentreX + newCentreY * newCentreY) / stat->m_SHOT_SPEED;
	long loop = 2;
	while(--loop >= 0) {
		double A = m_Speed.m_x * m_Speed.m_x + m_Speed.m_y * m_Speed.m_y - stat->m_SHOT_SPEED * stat->m_SHOT_SPEED;
		if(A != 0.0) {
			double x = newCentreX, y = newCentreY;
			double B = 2 * x * m_Speed.m_x + 2 * y * m_Speed.m_y;
			double C = x * x + y * y;
			double p = B / A, q = C / A;
			double det = p * p / 4 - q;
			if(det > 0) {
				double t1 = -p/2 + sqrt(det);
				double t2 = -p/2 - sqrt(det);
				if( (t1 >= 0) && (t2 >= 0) ) {
					if(t1 <= t2) {
						time2Hit = t1;
					}
					else {
						time2Hit = t2;
					}
				}
				else if(t1 > 0) {
					time2Hit = t1;
				}
				else if(t2 > 0) {
					time2Hit = t2;
				}
			}
			else if(det == 0) {
				double t1 = -p/2;
				if(t1 > 0) {
					time2Hit = t1;
				}
			}
			aimPoint.m_x = newCentreX + time2Hit * m_Speed.m_x;
			aimPoint.m_y = newCentreY + time2Hit * m_Speed.m_y;
			if(loop > 0) {
				bool finished = true; // set true: if a check fails set to false
				if(aimPoint.m_x > 511) {
					newCentreX -= 1024;
					finished = false;
				}
				else if(aimPoint.m_x < -512) {
					newCentreX += 1024;
					finished = false;
				}
				if(aimPoint.m_y > 383) {
					newCentreY -= 768;
					finished = false;
				}
				else if(aimPoint.m_y < -384) {
					newCentreY += 768;
					finished = false;
				}
				if(finished) {
					break;
				}
			}
		}
	}
	distance2Aim = sqrt(aimPoint.m_x*aimPoint.m_x + aimPoint.m_y*aimPoint.m_y);

	// angle to estimated aim point
	aimAngle = 180.0 * atan2(aimPoint.m_y, aimPoint.m_x) / M_PI;
	while(aimAngle < 0.0) aimAngle += 360;
	while(aimAngle > 360.0) aimAngle -= 360;
	// difference angle to current shot angle
	aimAngleOffset = aimAngle - shotAngle;
	while(aimAngleOffset > 180) aimAngleOffset -= 360;
	while(aimAngleOffset < -180) aimAngleOffset += 360;
}

bool ShootableObject::EstimateRealHitTime(double shotAngle) {
    // have to look that far in the future because the received info is older
    // always 1 frame in the future because current position of objects moved ahead
    long timeOffset = 0; 
	static Statistics *stat = Statistics::GetStatistics();
    {
		MyAutoLockCS al(stat->m_DataLock);
        timeOffset += stat->m_MissingFrames;
    }

	// first check if we can shot at this again
	if( (m_EstimatedBlowTime != 0) && (m_Shots >= m_MaxShots) ) {
        if(m_EstimatedBlowTime < m_TimeStamp) {
    		// reset -> we can use this as target again!
	    	m_Shots = 0;
		    m_EstimatedBlowTime = 0;
        }
        else {
            // nope ignore for now!
            //return false;
        }
	}

    double aimAngles[200];
    double distance2Aim[200];
    double time2Hit[200];
    const int MAXCOUNT = 200;
    Point aimPoint;
    double aimAngleOffset;
    long i;
    for(i = 0; i < MAXCOUNT; ++i) {
    	CalcAimPoint(shotAngle, i+timeOffset, aimPoint, distance2Aim[i], aimAngles[i], aimAngleOffset, time2Hit[i]);
    }
    bool counterClockWise = true;
    if(aimAngles[0] < aimAngles[1]) {
        double newAimAngles[200];
        newAimAngles[0] = aimAngles[0];
        double offset = 0;
        for(i = 1; i < MAXCOUNT; ++i) {
            if(aimAngles[i-1] > aimAngles[i]) {
                offset += 360.0;
            }
            newAimAngles[i] = aimAngles[i] + offset;
        }
        memmove(aimAngles, newAimAngles, 200 * sizeof(double) );
    }
    else {
        counterClockWise = false;
        double newAimAngles[200];
        newAimAngles[0] = aimAngles[0];
        double offset = 0;
        for(i = 1; i < MAXCOUNT; ++i) {
            if(aimAngles[i-1] < aimAngles[i]) {
                offset -= 360.0;
            }
            newAimAngles[i] = aimAngles[i] + offset;
        }
        memmove(aimAngles, newAimAngles, 200 * sizeof(double) );
    }

    double shotAngleTmp = shotAngle;
    if( shotAngleTmp > aimAngles[0]) shotAngleTmp -= 360.0;
	double angleRange;
    for( i = 0; i < MAXCOUNT; ++i, shotAngleTmp += 135.0 / 32.0) {
        angleRange = 180.0 * atan2(m_Size, distance2Aim[i]) / M_PI;
        if( shotAngleTmp > (aimAngles[i] - angleRange) ) break;
    }
    double deltaTLeft = i;
    // no direct hit: have to wait for astro!
    if( (i < MAXCOUNT) && (fabs(shotAngleTmp - aimAngles[i]) > angleRange) ) {
        if(!counterClockWise) {
            --i;
            deltaTLeft = i;
            shotAngleTmp -= 135.0 / 32.0;
            if(i < 0) i = 0;
        }
        for( ; i < MAXCOUNT; ++i) {
            if( fabs(shotAngleTmp - aimAngles[i]) < angleRange) {
                break;
            }
        }
    }
    long waitTLeft = i;

    shotAngleTmp = shotAngle;
    if( shotAngleTmp < aimAngles[0]) shotAngleTmp += 360.0;
    for( i = 0; i < MAXCOUNT; ++i, shotAngleTmp -= 135.0 / 32.0) {
        angleRange = 180.0 * atan2(m_Size, distance2Aim[i]) / M_PI;
        if( shotAngleTmp < (aimAngles[i] + angleRange) ) break;
    }
    double deltaTRight = i;
    // no direct hit: have to wait for astro!
    if( (i < MAXCOUNT) && (fabs(shotAngleTmp - aimAngles[i]) > angleRange) ) {
        if(counterClockWise) {
            --i;
            deltaTRight = i;
            shotAngleTmp += 135.0 / 32.0;
            if(i < 0) i = 0;
        }
        for( ; i < MAXCOUNT; ++i) {
            if( fabs(shotAngleTmp - aimAngles[i]) < angleRange) {
                break;
            }
        }
    }
    long waitTRight = i;

    m_Time2TurnLast = m_Time2Turn;
    m_Time2WaitLast = m_Time2Wait;
    m_Time2HitLast = m_Time2Hit;
    m_FasterTurnToHitLast = m_FasterTurnToHit;

    if( (waitTLeft == MAXCOUNT) && (waitTRight == MAXCOUNT) ) {
		m_AimAngleOffset = m_AimAngle = 0;
		m_FasterTurnToHit = 0;
        m_Distance2Aim = 9999;
		m_Time2Turn = 9999;
        m_Time2Wait = 9999;
		m_Time2Hit = 9999;
        return false;
    }
    else if((waitTLeft != MAXCOUNT) && (waitTRight != MAXCOUNT) ) {
        if( (waitTLeft + time2Hit[waitTLeft]) < (waitTRight + time2Hit[waitTRight]) ) {
    	    CalcAimPoint(shotAngle, 0, m_AimPoint, m_Distance2Aim, m_AimAngle, m_AimAngleOffset, m_Time2Hit);
		    if(m_AimAngle < shotAngle) m_AimAngle += 360.0;
		    m_AimAngleOffset = m_AimAngle - shotAngle;
            m_FasterTurnToHit = (deltaTLeft == 0)? 0: 1;
            m_Time2Wait = waitTLeft - deltaTLeft;
		    m_Time2Turn = deltaTLeft;
            m_Time2Hit = time2Hit[waitTLeft];
        }
        else {
    	    CalcAimPoint(shotAngle, 0, m_AimPoint, m_Distance2Aim, m_AimAngle, m_AimAngleOffset, m_Time2Hit);
		    if(m_AimAngle > shotAngle) m_AimAngle -= 360.0;
		    m_AimAngleOffset = m_AimAngle - shotAngle;
            m_FasterTurnToHit = (deltaTRight == 0)? 0: -1;
            m_Time2Wait = waitTRight - deltaTRight;
		    m_Time2Turn = deltaTRight;
            m_Time2Hit = time2Hit[waitTRight];
        }
    }
    else if(waitTLeft != MAXCOUNT) {
        CalcAimPoint(shotAngle, 0, m_AimPoint, m_Distance2Aim, m_AimAngle, m_AimAngleOffset, m_Time2Hit);
        if(m_AimAngle < shotAngle) m_AimAngle += 360.0;
        m_AimAngleOffset = m_AimAngle - shotAngle;
        m_FasterTurnToHit = (deltaTLeft == 0)? 0: 1;
        m_Time2Wait = waitTLeft - deltaTLeft;
        m_Time2Turn = deltaTLeft;
        m_Time2Hit = time2Hit[waitTLeft];
    }
    else {
	    CalcAimPoint(shotAngle, 0, m_AimPoint, m_Distance2Aim, m_AimAngle, m_AimAngleOffset, m_Time2Hit);
	    if(m_AimAngle > shotAngle) m_AimAngle -= 360.0;
	    m_AimAngleOffset = m_AimAngle - shotAngle;
        m_FasterTurnToHit = (deltaTRight == 0)? 0: -1;
        m_Time2Wait = waitTRight - deltaTRight;
	    m_Time2Turn = deltaTRight;
        m_Time2Hit = time2Hit[waitTRight];
    }
    return true;
}

/*****************************************************************************/
/*                                                                           */
/*                 A S T E R O I D                                           */
/*                                                                           */
/*****************************************************************************/
MyAsteroid::MyAsteroid() 
: m_SubType(0)
{
    m_Size = 32;
}

MyAsteroid::MyAsteroid(double ts, const Asteroid &a) 
: ShootableObject(ASTERIOD_ANY, ts, Point(a.m_x, a.m_y), a.m_sf), m_SubType(a.m_type)
{
	switch (m_Sf) {
		case 0:  // groer Asteroid
			m_Typ = ASTERIOD_BIG;
			m_Size = 33; //32; 
			break;
		case 15: // mittlerer Asteroid
			m_Typ = ASTERIOD_MEDIUM;
			m_Size = 18; //16;
			break;
		case 14: // kleiner Asteroid
			m_Typ = ASTERIOD_SMALL;
			m_Size = 8; // 10.5;
			break;
	}
}

// look if estimated position and given position / type / size match: if so update object
bool MyAsteroid::UpdateMatchingObject(double ts, ObjectType typ, int x, int y, int scaleFactor, int subType) {
	// check subtype
	bool bret = (m_SubType == subType);
	if(bret) {
		// match: check other things
		bret = ShootableObject::UpdateMatchingObject(ts, typ, x, y, scaleFactor, subType);
	}
	return bret;
}


/*****************************************************************************/
/*                                                                           */
/*                 U F O                                                     */
/*                                                                           */
/*****************************************************************************/
MyAsteroid::MyAsteroid(double ts, int x, int y, int sf) 
: ShootableObject(UFO, ts, Point(x, y), sf), m_SubType(5)
{
	switch (m_Sf) {
		case 15: // groes UFO
			m_Size = 20;
			break;
		case 14: // kleines UFO
			m_Size = 9;
			break;
	}
}


/*****************************************************************************/
/*                                                                           */
/*                 S H O T                                                   */
/*                                                                           */
/*****************************************************************************/
MyShot::MyShot() 
: m_ShotAngleIndex(-1), m_TargetID(-1), m_Time2Live(0.0), m_ShouldHit(0)
{
}

MyShot::MyShot(double ts, const Shot &s, long shotAngleIndex) 
: MovingObjectBase(SHOT, ts, Point(s.m_x, s.m_y)), m_ShotAngleIndex(shotAngleIndex), m_TargetID(-1),
  m_Time2Live(0.0), m_ShouldHit(0)
{
	if( (s.m_x * s.m_x + s.m_y * s.m_y) < 800 ) {
		m_Typ = FF; // friendly fire
	}
}

void MyShot::Update(double ts, int x, int y) {
    m_Time2Live -= (ts - m_TimeStamp);
    MovingObjectBase::Update(ts, x, y);
}
